<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Redirect if already logged in
if (isLoggedIn()) {
  $redirect = hasRole('employee') ? '/employee/dashboard.php' : '/company/dashboard.php';
  header("Location: " . BASE_URL . $redirect);
  exit();
}

$error = '';
$success = '';

// Check for timeout
if (isset($_GET['timeout'])) {
  $error = 'Your session has expired. Please login again.';
}

// Check for logout
if (isset($_GET['logout'])) {
  $success = 'You have been logged out successfully.';
}

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $email = sanitizeInput($_POST['email'] ?? '');
  $password = $_POST['password'] ?? '';
  $csrf_token = $_POST['csrf_token'] ?? '';

  // Validate CSRF token
  if (!verifyCSRFToken($csrf_token)) {
    $error = 'Invalid security token. Please try again.';
  }
  // Validate inputs
  elseif (empty($email) || empty($password)) {
    $error = 'Please fill in all fields.';
  }
  // Validate email format
  elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $error = 'Invalid email format.';
  } else {
    try {
      // Check if user exists
      $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ?");
      $stmt->execute([$email]);
      $user = $stmt->fetch();

      if (!$user) {
        // Email not found
        $error = '❌ Email not found: This email address is not registered in our system.';
      } elseif (!password_verify($password, $user['password'])) {
        // Password incorrect
        $error = '🔒 Incorrect password: The password you entered is incorrect. Please try again.';
      } else {
        // Login successful
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['role'] = $user['role'];
        $_SESSION['full_name'] = $user['full_name'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['LAST_ACTIVITY'] = time();

        // Regenerate session ID for security
        session_regenerate_id(true);

        // Redirect based on role
        $redirect = ($user['role'] === 'employee') ? '/employee/dashboard.php' : '/company/dashboard.php';
        header("Location: " . BASE_URL . $redirect);
        exit();
      }
    } catch (PDOException $e) {
      $error = 'An error occurred. Please try again later.';
    }
  }
}

$pageTitle = 'Login';
require_once 'includes/header.php';
?>

<div class="login-container">
  <div class="login-card">
    <div class="login-logo">
      <i class="bi bi-wallet2"></i>
      <h1><?php echo APP_NAME; ?></h1>
      <p class="text-muted">Professional Salary Management System</p>
    </div>

    <?php if ($error): ?>
      <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        <?php echo htmlspecialchars($error); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
    <?php endif; ?>

    <?php if ($success): ?>
      <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="bi bi-check-circle-fill me-2"></i>
        <?php echo htmlspecialchars($success); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
    <?php endif; ?>

    <form method="POST" action="" id="loginForm">
      <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">

      <div class="mb-3">
        <label for="email" class="form-label">
          <i class="bi bi-envelope"></i> Email Address
        </label>
        <input type="email" class="form-control" id="email" name="email"
          placeholder="Enter your email" required
          value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
      </div>

      <div class="mb-3">
        <label for="password" class="form-label">
          <i class="bi bi-lock"></i> Password
        </label>
        <div class="input-group">
          <input type="password" class="form-control" id="password" name="password"
            placeholder="Enter your password" required>
          <button class="btn btn-outline-secondary" type="button" id="togglePassword">
            <i class="bi bi-eye"></i>
          </button>
        </div>
      </div>

      <div class="mb-3 form-check">
        <input type="checkbox" class="form-check-input" id="remember">
        <label class="form-check-label" for="remember">
          Remember me
        </label>
      </div>

      <button type="submit" class="btn btn-primary w-100 mb-3">
        <i class="bi bi-box-arrow-in-right"></i> Login
      </button>

      <hr>

      <div class="text-center text-muted small">
        <p class="mb-1"><strong>Default Credentials:</strong></p>
        <p class="mb-1">Admin: admin@company.com / password</p>
        <p class="mb-0">Employee: john@employee.com / password</p>
      </div>
    </form>
  </div>
</div>

<script>
  // Toggle password visibility
  document.getElementById('togglePassword').addEventListener('click', function() {
    const passwordInput = document.getElementById('password');
    const icon = this.querySelector('i');

    if (passwordInput.type === 'password') {
      passwordInput.type = 'text';
      icon.classList.remove('bi-eye');
      icon.classList.add('bi-eye-slash');
    } else {
      passwordInput.type = 'password';
      icon.classList.remove('bi-eye-slash');
      icon.classList.add('bi-eye');
    }
  });

  // Form validation
  document.getElementById('loginForm').addEventListener('submit', function(e) {
    const email = document.getElementById('email').value;
    const password = document.getElementById('password').value;

    if (!email || !password) {
      e.preventDefault();
      alert('Please fill in all fields');
      return false;
    }

    // Show loading state
    const submitBtn = this.querySelector('button[type="submit"]');
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Logging in...';
    submitBtn.disabled = true;
  });
</script>

<?php require_once 'includes/footer.php'; ?>