<?php

/**
 * Common Functions
 * PayTrack - Professional Salary Management System
 */

// Get total payments for an employee
function getTotalPayments($employeeId)
{
  global $pdo;
  $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount), 0) as total FROM payments WHERE employee_id = ?");
  $stmt->execute([$employeeId]);
  return $stmt->fetch()['total'];
}

// Get pending amount for an employee
function getPendingAmount($employeeId)
{
  global $pdo;

  // Get employee info
  $stmt = $pdo->prepare("SELECT joining_date, monthly_salary FROM employees WHERE id = ?");
  $stmt->execute([$employeeId]);
  $employee = $stmt->fetch();

  if (!$employee) return 0;

  // Calculate months worked
  $joiningDate = new DateTime($employee['joining_date']);
  $currentDate = new DateTime();
  $interval = $joiningDate->diff($currentDate);
  $monthsWorked = ($interval->y * 12) + $interval->m + 1; // +1 for current month

  // Calculate total earned
  $totalEarned = $monthsWorked * $employee['monthly_salary'];

  // Get total received
  $totalReceived = getTotalPayments($employeeId);

  // Calculate pending
  $pending = $totalEarned - $totalReceived;

  return max(0, $pending); // Don't return negative
}

// Get total earned for an employee
function getTotalEarned($employeeId)
{
  global $pdo;

  $stmt = $pdo->prepare("SELECT joining_date, monthly_salary FROM employees WHERE id = ?");
  $stmt->execute([$employeeId]);
  $employee = $stmt->fetch();

  if (!$employee) return 0;

  $joiningDate = new DateTime($employee['joining_date']);
  $currentDate = new DateTime();
  $interval = $joiningDate->diff($currentDate);
  $monthsWorked = ($interval->y * 12) + $interval->m + 1;

  return $monthsWorked * $employee['monthly_salary'];
}

// Get recent payments
function getRecentPayments($employeeId, $limit = 5)
{
  global $pdo;
  $stmt = $pdo->prepare("
        SELECT p.*, u.full_name as added_by_name 
        FROM payments p 
        LEFT JOIN users u ON p.added_by = u.id 
        WHERE p.employee_id = ? 
        ORDER BY p.payment_date DESC 
        LIMIT ?
    ");
  $stmt->execute([$employeeId, $limit]);
  return $stmt->fetchAll();
}

// Get all payments for an employee
function getAllPayments($employeeId, $offset = 0, $limit = 10)
{
  global $pdo;
  $stmt = $pdo->prepare("
        SELECT p.*, u.full_name as added_by_name 
        FROM payments p 
        LEFT JOIN users u ON p.added_by = u.id 
        WHERE p.employee_id = ? 
        ORDER BY p.payment_date DESC 
        LIMIT ? OFFSET ?
    ");
  $stmt->execute([$employeeId, $limit, $offset]);
  return $stmt->fetchAll();
}

// Count total payments
function countPayments($employeeId)
{
  global $pdo;
  $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM payments WHERE employee_id = ?");
  $stmt->execute([$employeeId]);
  return $stmt->fetch()['count'];
}

// Get monthly breakdown
function getMonthlyBreakdown($employeeId)
{
  global $pdo;

  $stmt = $pdo->prepare("SELECT joining_date, monthly_salary FROM employees WHERE id = ?");
  $stmt->execute([$employeeId]);
  $employee = $stmt->fetch();

  if (!$employee) return [];

  $joiningDate = new DateTime($employee['joining_date']);
  $currentDate = new DateTime();
  $monthsWorked = ($currentDate->diff($joiningDate)->y * 12) + $currentDate->diff($joiningDate)->m + 1;

  $breakdown = [];
  $date = clone $joiningDate;

  for ($i = 0; $i < $monthsWorked; $i++) {
    $monthKey = $date->format('Y-m');

    // Get payments for this month
    $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(amount), 0) as paid 
            FROM payments 
            WHERE employee_id = ? 
            AND DATE_FORMAT(payment_date, '%Y-%m') = ?
        ");
    $stmt->execute([$employeeId, $monthKey]);
    $paid = $stmt->fetch()['paid'];

    $expected = $employee['monthly_salary'];
    $balance = $expected - $paid;

    $status = 'pending';
    if ($paid >= $expected) {
      $status = 'paid';
    } elseif ($paid > 0) {
      $status = 'partial';
    }

    $breakdown[] = [
      'month' => $date->format('M Y'),
      'month_key' => $monthKey,
      'expected' => $expected,
      'paid' => $paid,
      'balance' => max(0, $balance),
      'status' => $status
    ];

    $date->modify('+1 month');
  }

  return array_reverse($breakdown);
}

// Get payment method label
function getPaymentMethodLabel($method)
{
  $labels = [
    'bank_transfer' => 'Bank Transfer',
    'cash' => 'Cash',
    'cheque' => 'Cheque',
    'online' => 'Online Payment'
  ];
  return $labels[$method] ?? $method;
}

// Get status badge HTML
function getStatusBadge($status)
{
  $badges = [
    'paid' => '<span class="badge bg-success">Paid</span>',
    'partial' => '<span class="badge bg-warning">Partial</span>',
    'pending' => '<span class="badge bg-danger">Pending</span>',
    'active' => '<span class="badge bg-success">Active</span>',
    'inactive' => '<span class="badge bg-secondary">Inactive</span>'
  ];
  return $badges[$status] ?? '<span class="badge bg-secondary">' . $status . '</span>';
}

// Get all employees (for company)
function getAllEmployees()
{
  global $pdo;
  $stmt = $pdo->query("
        SELECT e.*, u.full_name, u.email, u.profile_picture 
        FROM employees e 
        LEFT JOIN users u ON e.user_id = u.id 
        ORDER BY e.created_at DESC
    ");
  return $stmt->fetchAll();
}

// Get employee by ID
function getEmployeeById($employeeId)
{
  global $pdo;
  $stmt = $pdo->prepare("
        SELECT e.*, u.full_name, u.email, u.profile_picture 
        FROM employees e 
        LEFT JOIN users u ON e.user_id = u.id 
        WHERE e.id = ?
    ");
  $stmt->execute([$employeeId]);
  return $stmt->fetch();
}

// Get company statistics
function getCompanyStats()
{
  global $pdo;

  // Total employees
  $stmt = $pdo->query("SELECT COUNT(*) as count FROM employees WHERE status = 'active'");
  $totalEmployees = $stmt->fetch()['count'];

  // Total pending
  $stmt = $pdo->query("SELECT e.id FROM employees e WHERE e.status = 'active'");
  $employees = $stmt->fetchAll();
  $totalPending = 0;
  foreach ($employees as $emp) {
    $totalPending += getPendingAmount($emp['id']);
  }

  // Payments this month
  $stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM payments 
        WHERE MONTH(payment_date) = MONTH(CURRENT_DATE()) 
        AND YEAR(payment_date) = YEAR(CURRENT_DATE())
    ");
  $stmt->execute();
  $paymentsThisMonth = $stmt->fetch()['count'];

  // Total disbursed
  $stmt = $pdo->query("SELECT COALESCE(SUM(amount), 0) as total FROM payments");
  $totalDisbursed = $stmt->fetch()['total'];

  return [
    'total_employees' => $totalEmployees,
    'total_pending' => $totalPending,
    'payments_this_month' => $paymentsThisMonth,
    'total_disbursed' => $totalDisbursed
  ];
}

// Validate password strength
function validatePassword($password)
{
  $errors = [];

  if (strlen($password) < 8) {
    $errors[] = "Password must be at least 8 characters long";
  }
  if (!preg_match('/[A-Z]/', $password)) {
    $errors[] = "Password must contain at least one uppercase letter";
  }
  if (!preg_match('/[a-z]/', $password)) {
    $errors[] = "Password must contain at least one lowercase letter";
  }
  if (!preg_match('/[0-9]/', $password)) {
    $errors[] = "Password must contain at least one number";
  }

  return $errors;
}

// Show alert message
function showAlert($type, $message)
{
  return '<div class="alert alert-' . $type . ' alert-dismissible fade show" role="alert">
                ' . htmlspecialchars($message) . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
}
