<?php
require_once '../config/config.php';
require_once '../includes/functions.php';

requireRole('employee');

$userInfo = getUserInfo($_SESSION['user_id']);
$employeeInfo = getEmployeeInfo($_SESSION['user_id']);

$success = '';
$error = '';

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
  $fullName = sanitizeInput($_POST['full_name']);
  $email = sanitizeInput($_POST['email']);

  if (empty($fullName) || empty($email)) {
    $error = 'Please fill in all required fields.';
  } else {
    try {
      $stmt = $pdo->prepare("UPDATE users SET full_name = ?, email = ? WHERE id = ?");
      $stmt->execute([$fullName, $email, $_SESSION['user_id']]);
      $success = 'Profile updated successfully!';
      $_SESSION['full_name'] = $fullName;
      $userInfo = getUserInfo($_SESSION['user_id']);
    } catch (PDOException $e) {
      $error = 'Error updating profile: ' . $e->getMessage();
    }
  }
}

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
  $currentPassword = $_POST['current_password'];
  $newPassword = $_POST['new_password'];
  $confirmPassword = $_POST['confirm_password'];

  if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
    $error = 'Please fill in all password fields.';
  } elseif ($newPassword !== $confirmPassword) {
    $error = 'New passwords do not match.';
  } elseif (!password_verify($currentPassword, $userInfo['password'])) {
    $error = 'Current password is incorrect.';
  } else {
    $passwordErrors = validatePassword($newPassword);
    if (!empty($passwordErrors)) {
      $error = implode('<br>', $passwordErrors);
    } else {
      try {
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->execute([$hashedPassword, $_SESSION['user_id']]);
        $success = 'Password changed successfully!';
      } catch (PDOException $e) {
        $error = 'Error changing password: ' . $e->getMessage();
      }
    }
  }
}

$pageTitle = 'Profile';
require_once '../includes/header.php';
require_once '../includes/navbar.php';
?>

<div class="container-fluid py-4">
  <div class="row mb-4">
    <div class="col-12">
      <nav aria-label="breadcrumb">
        <ol class="breadcrumb">
          <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
          <li class="breadcrumb-item active">Profile</li>
        </ol>
      </nav>
    </div>
  </div>

  <?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <?php echo $success; ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <?php echo $error; ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <div class="row">
    <!-- Profile Information -->
    <div class="col-md-6 mb-4">
      <div class="card">
        <div class="card-header">
          <i class="bi bi-person"></i> Personal Information
        </div>
        <div class="card-body">
          <form method="POST">
            <input type="hidden" name="update_profile" value="1">

            <div class="mb-3">
              <label class="form-label">Full Name</label>
              <input type="text" class="form-control" name="full_name"
                value="<?php echo htmlspecialchars($userInfo['full_name']); ?>" required>
            </div>

            <div class="mb-3">
              <label class="form-label">Email Address</label>
              <input type="email" class="form-control" name="email"
                value="<?php echo htmlspecialchars($userInfo['email']); ?>" required>
            </div>

            <div class="mb-3">
              <label class="form-label">Employee ID</label>
              <input type="text" class="form-control"
                value="<?php echo htmlspecialchars($employeeInfo['employee_id']); ?>" disabled>
            </div>

            <button type="submit" class="btn btn-primary">
              <i class="bi bi-save"></i> Update Profile
            </button>
          </form>
        </div>
      </div>
    </div>

    <!-- Change Password -->
    <div class="col-md-6 mb-4">
      <div class="card">
        <div class="card-header">
          <i class="bi bi-lock"></i> Change Password
        </div>
        <div class="card-body">
          <form method="POST">
            <input type="hidden" name="change_password" value="1">

            <div class="mb-3">
              <label class="form-label">Current Password</label>
              <input type="password" class="form-control" name="current_password" required>
            </div>

            <div class="mb-3">
              <label class="form-label">New Password</label>
              <input type="password" class="form-control" name="new_password" required>
              <small class="text-muted">Min 8 characters, including uppercase, lowercase, and number</small>
            </div>

            <div class="mb-3">
              <label class="form-label">Confirm New Password</label>
              <input type="password" class="form-control" name="confirm_password" required>
            </div>

            <button type="submit" class="btn btn-warning">
              <i class="bi bi-key"></i> Change Password
            </button>
          </form>
        </div>
      </div>
    </div>
  </div>

  <!-- Employment Details (Read-only) -->
  <div class="row">
    <div class="col-12">
      <div class="card">
        <div class="card-header">
          <i class="bi bi-briefcase"></i> Employment Details
        </div>
        <div class="card-body">
          <div class="row">
            <div class="col-md-3 mb-3">
              <label class="text-muted">Company Name</label>
              <p class="mb-0"><strong><?php echo htmlspecialchars($employeeInfo['company_name']); ?></strong></p>
            </div>
            <div class="col-md-3 mb-3">
              <label class="text-muted">Designation</label>
              <p class="mb-0"><strong><?php echo htmlspecialchars($employeeInfo['designation']); ?></strong></p>
            </div>
            <div class="col-md-3 mb-3">
              <label class="text-muted">Department</label>
              <p class="mb-0"><strong><?php echo htmlspecialchars($employeeInfo['department']); ?></strong></p>
            </div>
            <div class="col-md-3 mb-3">
              <label class="text-muted">Status</label>
              <p class="mb-0"><?php echo getStatusBadge($employeeInfo['status']); ?></p>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<?php require_once '../includes/footer.php'; ?>