<?php

/**
 * Application Configuration
 * PayTrack - Professional Salary Management System
 */

// Start session with secure settings
if (session_status() === PHP_SESSION_NONE) {
  ini_set('session.cookie_httponly', 1);
  ini_set('session.use_only_cookies', 1);
  ini_set('session.cookie_secure', 0); // Set to 1 if using HTTPS
  session_start();
}

// Application settings
define('APP_NAME', 'PayTrack');
define('APP_VERSION', '1.0.0');
define('BASE_URL', 'http://localhost:8080'); // Update this based on your setup

// File upload settings
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_EXTENSIONS', ['jpg', 'jpeg', 'png', 'pdf']);
define('UPLOAD_PATH', __DIR__ . '/../assets/uploads/');

// Session timeout (30 minutes)
define('SESSION_TIMEOUT', 1800);

// Pagination
define('ITEMS_PER_PAGE', 10);

// Date format
define('DATE_FORMAT', 'Y-m-d');
define('DISPLAY_DATE_FORMAT', 'd M, Y');
define('DATETIME_FORMAT', 'Y-m-d H:i:s');

// Currency
define('CURRENCY_SYMBOL', 'PKR');
define('CURRENCY_CODE', 'PKR');

// Timezone
date_default_timezone_set('Asia/Karachi');

// Error reporting (disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include database connection
require_once __DIR__ . '/database.php';

// Session timeout check
function checkSessionTimeout()
{
  if (isset($_SESSION['LAST_ACTIVITY']) && (time() - $_SESSION['LAST_ACTIVITY'] > SESSION_TIMEOUT)) {
    session_unset();
    session_destroy();
    header("Location: " . BASE_URL . "/login.php?timeout=1");
    exit();
  }
  $_SESSION['LAST_ACTIVITY'] = time();
}

// Generate CSRF token
function generateCSRFToken()
{
  if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
  }
  return $_SESSION['csrf_token'];
}

// Verify CSRF token
function verifyCSRFToken($token)
{
  return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Check if user is logged in
function isLoggedIn()
{
  return isset($_SESSION['user_id']) && isset($_SESSION['role']);
}

// Check if user has specific role
function hasRole($role)
{
  return isLoggedIn() && $_SESSION['role'] === $role;
}

// Redirect if not logged in
function requireLogin()
{
  if (!isLoggedIn()) {
    header("Location: " . BASE_URL . "/login.php");
    exit();
  }
  checkSessionTimeout();
}

// Redirect if not specific role
function requireRole($role)
{
  requireLogin();
  if (!hasRole($role)) {
    header("Location: " . BASE_URL . "/login.php");
    exit();
  }
}

// Sanitize input
function sanitizeInput($data)
{
  $data = trim($data);
  $data = stripslashes($data);
  $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
  return $data;
}

// Format currency
function formatCurrency($amount)
{
  return CURRENCY_SYMBOL . ' ' . number_format($amount, 2);
}

// Format date
function formatDate($date, $format = DISPLAY_DATE_FORMAT)
{
  return date($format, strtotime($date));
}

// Calculate months between two dates
function monthsBetween($date1, $date2)
{
  $datetime1 = new DateTime($date1);
  $datetime2 = new DateTime($date2);
  $interval = $datetime1->diff($datetime2);
  return ($interval->y * 12) + $interval->m;
}

// Upload file
function uploadFile($file, $prefix = '')
{
  if ($file['error'] !== UPLOAD_ERR_OK) {
    return ['success' => false, 'message' => 'Upload error'];
  }

  if ($file['size'] > MAX_FILE_SIZE) {
    return ['success' => false, 'message' => 'File size exceeds limit'];
  }

  $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
  if (!in_array($ext, ALLOWED_EXTENSIONS)) {
    return ['success' => false, 'message' => 'Invalid file type'];
  }

  $filename = $prefix . '_' . uniqid() . '.' . $ext;
  $filepath = UPLOAD_PATH . $filename;

  if (move_uploaded_file($file['tmp_name'], $filepath)) {
    return ['success' => true, 'filename' => $filename];
  }

  return ['success' => false, 'message' => 'Failed to move uploaded file'];
}

// Get user info
function getUserInfo($userId)
{
  global $pdo;
  $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
  $stmt->execute([$userId]);
  return $stmt->fetch();
}

// Get employee info by user_id
function getEmployeeInfo($userId)
{
  global $pdo;
  $stmt = $pdo->prepare("SELECT * FROM employees WHERE user_id = ?");
  $stmt->execute([$userId]);
  return $stmt->fetch();
}
