<?php
require_once '../config/config.php';
require_once '../includes/functions.php';

requireRole('company');

$success = '';
$error = '';

// Handle Add Payment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['add_payment']) || isset($_POST['quick_payment']))) {
  $employeeId = sanitizeInput($_POST['employee_id']);
  $paymentDate = sanitizeInput($_POST['payment_date'] ?? date('Y-m-d'));
  $amount = sanitizeInput($_POST['amount']);
  $paymentMethod = sanitizeInput($_POST['payment_method']);
  $transactionId = sanitizeInput($_POST['transaction_id'] ?? '');
  $notes = sanitizeInput($_POST['notes'] ?? '');

  if (empty($employeeId) || empty($amount) || empty($paymentMethod)) {
    $error = 'Please fill all required fields.';
  } elseif ($amount <= 0) {
    $error = 'Amount must be greater than 0.';
  } else {
    try {
      $stmt = $pdo->prepare("INSERT INTO payments (employee_id, payment_date, amount, payment_method, transaction_id, notes, added_by) 
                                   VALUES (?, ?, ?, ?, ?, ?, ?)");
      $stmt->execute([$employeeId, $paymentDate, $amount, $paymentMethod, $transactionId, $notes, $_SESSION['user_id']]);
      $success = 'Payment added successfully!';
    } catch (PDOException $e) {
      $error = 'Error: ' . $e->getMessage();
    }
  }
}

// Handle Delete Payment
if (isset($_GET['delete'])) {
  try {
    $stmt = $pdo->prepare("DELETE FROM payments WHERE id = ?");
    $stmt->execute([$_GET['delete']]);
    $success = 'Payment deleted successfully!';
  } catch (PDOException $e) {
    $error = 'Error deleting payment: ' . $e->getMessage();
  }
}

// Get all payments
$query = "SELECT p.*, e.employee_id, e.monthly_salary, u.full_name, u2.full_name as added_by_name 
          FROM payments p 
          LEFT JOIN employees e ON p.employee_id = e.id 
          LEFT JOIN users u ON e.user_id = u.id
          LEFT JOIN users u2 ON p.added_by = u2.id
          ORDER BY p.payment_date DESC";
$stmt = $pdo->query($query);
$payments = $stmt->fetchAll();

$employees = getAllEmployees();

$pageTitle = 'Manage Payments';
require_once '../includes/header.php';
require_once '../includes/navbar.php';
?>

<div class="container-fluid py-4">
  <div class="row mb-4">
    <div class="col-12">
      <nav aria-label="breadcrumb">
        <ol class="breadcrumb">
          <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
          <li class="breadcrumb-item active">Manage Payments</li>
        </ol>
      </nav>
    </div>
  </div>

  <?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <?php echo $success; ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <?php echo $error; ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <div class="row">
    <div class="col-md-4 mb-4">
      <div class="card">
        <div class="card-header">
          <i class="bi bi-plus-circle"></i> Add New Payment
        </div>
        <div class="card-body">
          <form method="POST">
            <input type="hidden" name="add_payment" value="1">

            <div class="mb-3">
              <label class="form-label">Employee *</label>
              <select class="form-select" name="employee_id" required>
                <option value="">Select Employee</option>
                <?php foreach ($employees as $emp): ?>
                  <?php if ($emp['status'] === 'active'): ?>
                    <option value="<?php echo $emp['id']; ?>"
                      <?php echo (isset($_GET['employee_id']) && $_GET['employee_id'] == $emp['id']) ? 'selected' : ''; ?>>
                      <?php echo htmlspecialchars($emp['full_name']); ?>
                      (<?php echo $emp['employee_id']; ?>)
                    </option>
                  <?php endif; ?>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="mb-3">
              <label class="form-label">Payment Date *</label>
              <input type="date" class="form-control" name="payment_date"
                value="<?php echo date('Y-m-d'); ?>" required>
            </div>

            <div class="mb-3">
              <label class="form-label">Amount (PKR) *</label>
              <input type="number" class="form-control" name="amount"
                step="0.01" min="0" required>
            </div>

            <div class="mb-3">
              <label class="form-label">Payment Method *</label>
              <select class="form-select" name="payment_method" required>
                <option value="bank_transfer">Bank Transfer</option>
                <option value="cash">Cash</option>
                <option value="cheque">Cheque</option>
                <option value="online">Online Payment</option>
              </select>
            </div>

            <div class="mb-3">
              <label class="form-label">Transaction ID</label>
              <input type="text" class="form-control" name="transaction_id">
            </div>

            <div class="mb-3">
              <label class="form-label">Notes</label>
              <textarea class="form-control" name="notes" rows="3"></textarea>
            </div>

            <button type="submit" class="btn btn-success w-100">
              <i class="bi bi-check-circle"></i> Add Payment
            </button>
          </form>
        </div>
      </div>
    </div>

    <div class="col-md-8 mb-4">
      <div class="card">
        <div class="card-header">
          <i class="bi bi-cash-stack"></i> All Payments
        </div>
        <div class="card-body">
          <?php if (count($payments) > 0): ?>
            <div class="table-responsive">
              <table class="table table-hover data-table">
                <thead>
                  <tr>
                    <th>Date</th>
                    <th>Employee</th>
                    <th>Amount</th>
                    <th>Method</th>
                    <th>Transaction ID</th>
                    <th>Added By</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($payments as $payment): ?>
                    <tr>
                      <td><?php echo formatDate($payment['payment_date']); ?></td>
                      <td>
                        <strong><?php echo htmlspecialchars($payment['full_name']); ?></strong>
                        <br><small class="text-muted"><?php echo $payment['employee_id']; ?></small>
                      </td>
                      <td><strong class="text-success"><?php echo formatCurrency($payment['amount']); ?></strong></td>
                      <td>
                        <span class="badge bg-secondary">
                          <?php echo getPaymentMethodLabel($payment['payment_method']); ?>
                        </span>
                      </td>
                      <td><code><?php echo htmlspecialchars($payment['transaction_id'] ?? 'N/A'); ?></code></td>
                      <td><?php echo htmlspecialchars($payment['added_by_name']); ?></td>
                      <td>
                        <a href="?delete=<?php echo $payment['id']; ?>"
                          class="btn btn-sm btn-danger"
                          onclick="return confirm('Are you sure you want to delete this payment?')">
                          <i class="bi bi-trash"></i>
                        </a>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <div class="text-center py-5">
              <i class="bi bi-inbox display-1 text-muted"></i>
              <p class="text-muted mt-3">No payments recorded yet</p>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
</div>

<?php require_once '../includes/footer.php'; ?>