/**
 * PayTrack - Main JavaScript
 * Handles interactions, AJAX calls, and dynamic UI updates
 */

// Show toast notification
function showToast(message, type = 'success') {
  const toastContainer = document.querySelector('.toast-container');
  if (!toastContainer) return;

  const toastId = 'toast-' + Date.now();
  const bgClass = type === 'success' ? 'bg-success' :
    type === 'error' ? 'bg-danger' :
      type === 'warning' ? 'bg-warning' : 'bg-info';

  const toastHTML = `
        <div id="${toastId}" class="toast align-items-center text-white ${bgClass} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;

  toastContainer.insertAdjacentHTML('beforeend', toastHTML);

  const toastElement = document.getElementById(toastId);
  const toast = new bootstrap.Toast(toastElement, { autohide: true, delay: 3000 });
  toast.show();

  // Remove toast element after it's hidden
  toastElement.addEventListener('hidden.bs.toast', function () {
    toastElement.remove();
  });
}

// Confirm delete action
function confirmDelete(message = 'Are you sure you want to delete this?') {
  return confirm(message);
}

// Format currency input
document.querySelectorAll('input[type="number"][step="0.01"]').forEach(input => {
  input.addEventListener('blur', function () {
    if (this.value) {
      this.value = parseFloat(this.value).toFixed(2);
    }
  });
});

// Auto-dismiss alerts after 5 seconds
document.addEventListener('DOMContentLoaded', function () {
  const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
  alerts.forEach(alert => {
    setTimeout(() => {
      const bsAlert = new bootstrap.Alert(alert);
      bsAlert.close();
    }, 5000);
  });
});

// Form validation
document.querySelectorAll('form').forEach(form => {
  form.addEventListener('submit', function (e) {
    if (!this.checkValidity()) {
      e.preventDefault();
      e.stopPropagation();
    }
    this.classList.add('was-validated');
  });
});

// Smooth scroll
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
  anchor.addEventListener('click', function (e) {
    const href = this.getAttribute('href');
    if (href !== '#' && document.querySelector(href)) {
      e.preventDefault();
      document.querySelector(href).scrollIntoView({
        behavior: 'smooth'
      });
    }
  });
});

// Active nav link highlighting
const currentPage = window.location.pathname.split('/').pop();
document.querySelectorAll('.nav-link').forEach(link => {
  if (link.getAttribute('href') === currentPage) {
    link.classList.add('active');
  }
});

// Initialize tooltips
const tooltipTriggerList = document.querySelectorAll('[data-bs-toggle="tooltip"]');
const tooltipList = [...tooltipTriggerList].map(tooltipTriggerEl => new bootstrap.Tooltip(tooltipTriggerEl));

// Initialize popovers
const popoverTriggerList = document.querySelectorAll('[data-bs-toggle="popover"]');
const popoverList = [...popoverTriggerList].map(popoverTriggerEl => new bootstrap.Popover(popoverTriggerEl));

// Print functionality
function printContent() {
  window.print();
}

// Export to CSV (basic implementation)
function exportTableToCSV(tableId, filename = 'data.csv') {
  const table = document.getElementById(tableId);
  if (!table) return;

  let csv = [];
  const rows = table.querySelectorAll('tr');

  rows.forEach(row => {
    const cols = row.querySelectorAll('td, th');
    const rowData = Array.from(cols).map(col => {
      return '"' + col.innerText.replace(/"/g, '""') + '"';
    });
    csv.push(rowData.join(','));
  });

  const csvString = csv.join('\n');
  const blob = new Blob([csvString], { type: 'text/csv' });
  const url = window.URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  a.click();
  window.URL.revokeObjectURL(url);
}

// Loading spinner
function showLoading(button) {
  const originalText = button.innerHTML;
  button.setAttribute('data-original-text', originalText);
  button.disabled = true;
  button.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Loading...';
}

function hideLoading(button) {
  const originalText = button.getAttribute('data-original-text');
  button.disabled = false;
  button.innerHTML = originalText;
}

// Session timeout warning
let sessionTimeout;
let warningTimeout;

function resetSessionTimer() {
  clearTimeout(sessionTimeout);
  clearTimeout(warningTimeout);

  // Warn 2 minutes before timeout
  warningTimeout = setTimeout(() => {
    showToast('Your session will expire in 2 minutes due to inactivity.', 'warning');
  }, 28 * 60 * 1000); // 28 minutes

  // Timeout after 30 minutes
  sessionTimeout = setTimeout(() => {
    showToast('Your session has expired. Redirecting to login...', 'error');
    setTimeout(() => {
      window.location.href = '/PayTrack/login.php?timeout=1';
    }, 2000);
  }, 30 * 60 * 1000); // 30 minutes
}

// Reset timer on user activity
['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart', 'click'].forEach(event => {
  document.addEventListener(event, resetSessionTimer, true);
});

// Initialize session timer
if (document.querySelector('.navbar')) {
  resetSessionTimer();
}

// Password strength indicator
function checkPasswordStrength(password) {
  let strength = 0;

  if (password.length >= 8) strength++;
  if (password.length >= 12) strength++;
  if (/[a-z]/.test(password)) strength++;
  if (/[A-Z]/.test(password)) strength++;
  if (/[0-9]/.test(password)) strength++;
  if (/[^a-zA-Z0-9]/.test(password)) strength++;

  return strength;
}

// Add password strength indicator to password inputs
document.querySelectorAll('input[type="password"][name="new_password"], input[type="password"][name="password"]').forEach(input => {
  const strengthBar = document.createElement('div');
  strengthBar.className = 'progress mt-2';
  strengthBar.style.height = '5px';
  strengthBar.innerHTML = '<div class="progress-bar" role="progressbar"></div>';

  if (input.parentElement.classList.contains('input-group')) {
    input.parentElement.after(strengthBar);
  } else {
    input.after(strengthBar);
  }

  input.addEventListener('input', function () {
    const strength = checkPasswordStrength(this.value);
    const bar = strengthBar.querySelector('.progress-bar');
    const percentage = (strength / 6) * 100;

    bar.style.width = percentage + '%';

    if (percentage < 40) {
      bar.className = 'progress-bar bg-danger';
    } else if (percentage < 70) {
      bar.className = 'progress-bar bg-warning';
    } else {
      bar.className = 'progress-bar bg-success';
    }
  });
});

// Initialize DataTables
$(document).ready(function () {
  // Initialize all tables with class 'data-table'
  if ($.fn.DataTable) {
    $('.data-table').DataTable({
      responsive: true,
      pageLength: 10,
      lengthMenu: [[10, 25, 50, -1], [10, 25, 50, "All"]],
      language: {
        search: "Search:",
        lengthMenu: "Show _MENU_ entries",
        info: "Showing _START_ to _END_ of _TOTAL_ entries",
        paginate: {
          first: "First",
          last: "Last",
          next: "Next",
          previous: "Previous"
        }
      },
      order: [[0, 'desc']] // Default sort by first column descending
    });
  }
});

console.log('PayTrack initialized successfully!');
